<?php
/**
 * Access Levels
 *
 * @package     AutomatorWP\Restrict_Content\Custom_Tables\User_Levels
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Parse query args for fields
 *
 * @since 1.0.0
 *
 * @param string $where
 * @param CT_Query $ct_query
 *
 * @return string
 */
function automatorwp_restrict_content_access_levels_query_where( $where, $ct_query ) {

    global $ct_table;

    if( $ct_table->name !== 'automatorwp_restrict_content_access_levels' ) {
        return $where;
    }

    // Shorthand
    $qv = $ct_query->query_vars;

    // Name
    $where .= automatorwp_custom_table_where( $qv, 'name', 'string' );
    
    // Slug
    $where .= automatorwp_custom_table_where( $qv, 'sñug', 'string' );

    return $where;
}
add_filter( 'ct_query_where', 'automatorwp_restrict_content_access_levels_query_where', 10, 2 );

/**
 * Define the search fields for custom fields
 *
 * @since 1.0.0
 *
 * @param array $search_fields
 *
 * @return array
 */
function automatorwp_restrict_content_access_levels_search_fields( $search_fields ) {

    $search_fields[] = 'name';
    $search_fields[] = 'slug';

    return $search_fields;

}
add_filter( 'ct_query_automatorwp_restrict_content_access_levels_search_fields', 'automatorwp_restrict_content_access_levels_search_fields' );

/**
 * Columns for custom fields level view
 *
 * @since 1.0.0
 *
 * @param array $columns
 *
 * @return array
 */
function automatorwp_restrict_content_access_levels_manage_levels_columns( $columns = array() ) {

    $columns['name']          = __( 'Name', 'automatorwp-pro' );
    $columns['slug']          = __( 'Slug', 'automatorwp-pro' );
    $columns['users']         = __( 'Users', 'automatorwp-pro' );
    $columns['shortcode']     = __( 'Shortcode', 'automatorwp-pro' );

    return $columns;
}
add_filter( 'manage_automatorwp_restrict_content_access_levels_columns', 'automatorwp_restrict_content_access_levels_manage_levels_columns' );

/**
 * Sortable columns for custom fields level view
 *
 * @since 1.0.0
 *
 * @param array $sortable_columns
 *
 * @return array
 */
function automatorwp_restrict_content_access_levels_manage_levels_sortable_columns( $sortable_columns ) {

    $sortable_columns['name']               = array( 'name', false );
    $sortable_columns['slug']               = array( 'slug', false );

    return $sortable_columns;

}
add_filter( 'manage_automatorwp_restrict_content_access_levels_sortable_columns', 'automatorwp_restrict_content_access_levels_manage_levels_sortable_columns' );

/**
 * Columns rendering for custom fields level view
 *
 * @since  1.0.0
 *
 * @param string $column_name
 * @param integer $object_id
 */
function automatorwp_restrict_content_access_levels_manage_custom_column(  $column_name, $object_id ) {

    // Setup vars
    $access_level = ct_get_object( $object_id );

    switch( $column_name ) {
        case 'name':
            $name = $access_level->name;
            if( empty( $name ) ) {
                $name = __( '(no name)', 'automatorwp-pro' );
            }
            ?>
            <strong><a href="<?php echo ct_get_edit_link( 'automatorwp_restrict_content_access_levels', $access_level->id ); ?>"><?php echo $name; ?></a></strong>
            <?php

            break;
        case 'slug':
            $slug = $access_level->slug;

            if( empty( $slug ) ) {
                $slug = sanitize_key( $access_level->name );
            }

            if( empty( $slug ) ) {
                $slug = $access_level->id;
            }

            echo $slug;

            break;
        case 'users':
            $count = automatorwp_restrict_content_get_access_level_user_count( $access_level->id );
            ?>
            <?php echo $count; ?>
            <?php

            break;
        case 'shortcode':
            $shortcode = '[awp_rc access_level="' . $access_level->slug . '"]' . __( 'Content to show', 'automatorwp-pro' ) .  '[/awp_rc]';
            ?>
            <input value="<?php echo esc_attr($shortcode); ?>" onclick="this.focus(); this.select();" readonly class="automatorwp-restrict-content-shortcode-input"/>
            <?php
            break;
    }
}
add_action( 'manage_automatorwp_restrict_content_access_levels_custom_column', 'automatorwp_restrict_content_access_levels_manage_custom_column', 10, 2 );

/**
 * Default data when creating a new item (similar to WP auto draft) see ct_insert_object()
 *
 * @since  1.0.0
 *
 * @param array $default_data
 *
 * @return array
 */
function automatorwp_restrict_content_access_levels_default_data( $default_data = array() ) {

    $default_data['name'] = '';
    $default_data['slug'] = '';

    return $default_data;
}
add_filter( 'ct_automatorwp_restrict_content_access_levels_default_data', 'automatorwp_restrict_content_access_levels_default_data' );

/**
 * custom fields meta boxes
 *
 * @since  1.0.0
 */
function automatorwp_restrict_content_access_levels_add_meta_boxes() {

    add_meta_box( 'automatorwp_restrict_content_access_levels_name', __( 'Name', 'automatorwp-pro' ), 'automatorwp_restrict_content_access_levels_name_meta_box', 'automatorwp_restrict_content_access_levels', 'normal', 'default' );
    add_meta_box( 'automatorwp_restrict_content_access_levels_users', __( 'Users', 'automatorwp-pro' ), 'automatorwp_restrict_content_access_levels_users_meta_box', 'automatorwp_restrict_content_access_levels', 'normal', 'default' );
    add_meta_box( 'automatorwp_restrict_content_access_levels_shortcode', __( 'Shortcode', 'automatorwp-pro' ), 'automatorwp_restrict_content_access_levels_shortcode_meta_box', 'automatorwp_restrict_content_access_levels', 'side', 'default' );

}
add_action( 'add_meta_boxes', 'automatorwp_restrict_content_access_levels_add_meta_boxes' );

/**
 * Render the name meta box
 *
 * @since  1.0.0
 *
 * @param  stdClass $object The current object
 */
function automatorwp_restrict_content_access_levels_name_meta_box( $object = null ) {

    // Setup the CMB2 form
    $cmb2 = new CMB2( array(
        'id'        => 'automatorwp_restrict_content_access_levels_form',
        'object_types' => array( 'automatorwp_restrict_content_access_levels' ),
        'classes'   => 'automatorwp-form automatorwp-restrict-content-form',
        'hookup'    => false,
    ), $object->id );

    // Add the field to the form
    $cmb2->add_field( array(
        'id' => 'name',
        'name' => __('Name:', 'automatorwp-pro'),
        //'desc' => __('The field name.', 'automatorwp-pro'),
        'attributes' => array(
                'placeholder' => __('Enter access level name here', 'automatorwp-pro'),
        ),
        'type' => 'text',
    ) );

    // Add the field to the form
    $cmb2->add_field( array(
        'id' => 'slug',
        'name' => __('Slug:', 'automatorwp-pro'),
        //'desc' => __('The field name.', 'automatorwp-pro'),
        'attributes' => array(
            'type' => 'hidden',
        ),
        'type' => 'text',
        'after_field' => 'automatorwp_restrict_content_access_levels_before_slug_field',
    ) );

    // Render the form
    CMB2_Hookup::enqueue_cmb_css();
    CMB2_Hookup::enqueue_cmb_js();
    $cmb2->show_form();

    ?>

    <?php
}

function automatorwp_restrict_content_access_levels_before_slug_field( $field_args, $field ) {
    ?>
    <div class="automatorwp-restrict-content-user-levels-slug-preview">
        <strong><?php _e( 'Slug:', 'automatorwp-pro' ); ?></strong> <span class="automatorwp-restrict-content-user-levels-slug"><?php echo $field->escaped_value(); ?></span>
    </div>
    <?php
}

/**
 * Render the users meta box
 *
 * @since  1.0.0
 *
 * @param  stdClass $object The current object
 */
function automatorwp_restrict_content_access_levels_users_meta_box( $object = null ) {

    echo '<h3 class="automatorwp-restrict-content-access-levels-users-title">' . __( 'Users', 'automatorwp-pro' ) . '</h3>';

    ct_render_ajax_list_table( 'automatorwp_restrict_content_access_level_users',
        array(
            'access_level_id' => absint( $object->id ),
            'items_per_page' => 20,
        ),
        array(
            'views' => false,
            'search_box' => false // false until make search button work through ajax
        )
    );

    ?>
        <div class="automatorwp-form">
            <div class="automatorwp-user-selector">
                <select id="automatorwp-restrict-content-user-id"></select>
                <button class="button automatorwp-button-success automatorwp-restrict-content-add-user-button"><?php echo automatorwp_dashicon('plus') . ' ' . __( 'Add user to access level', 'automatorwp-pro' ); ?></button>
                <div class="automatorwp-access-level-add-user-message"></div>
            </div>
        </div>
    <?php
}

/**
 * Render the shortcode meta box
 *
 * @since  1.0.0
 *
 * @param  stdClass $object The current object
 */
function automatorwp_restrict_content_access_levels_shortcode_meta_box( $object = null ) {

    $access_level = ct_get_object( absint( $object->id ) );

    echo '<p class="automatorwp-restrict-content-access-levels-shortcode-desc">' . __( 'Use this shortcode on a page or post to create a section of restricted content that only gets shown to users in this access level.', 'automatorwp-pro' ) . '</p>';

    $shortcode = '[awp_rc access_level="' . $access_level->slug . '"]' . __( 'Content to show', 'automatorwp-pro' ) .  '[/awp_rc]';
    ?>
    <input value="<?php echo esc_attr($shortcode); ?>" onclick="this.focus(); this.select();" readonly class="automatorwp-restrict-content-shortcode-input"/>
    <?php
}